// EEPROM functions in general are here.  Some were too embedded in
// their functions to move here so remain elsewhere.
// Definition of registers is in the main file, along with #defines,
// globals, etc.

// *********** poly() ******************************************
//
// ***   Get frequency compensation polynomial coefficients ***
// ***   from user and store in EEPROM.  Also function to *****
// ***   read them back on start-up

void poly()
{
  float XC[6]; // temp place for values before accepting
  char choice;
  
  get_poly_from_ee();  // copy EEPROM to global
  for(int i = 0; i < 6; i++){
    XC[i] = C[i];      // copy global to temp
  }
  while(1){
    Serial.println(F("\xC0\r    POLYNOMIAL COEFFICIENTS\r")); //send CR+LF
    Serial.println(F("Make selection:\r"));
    display_XC(XC);       // show user his current values
    Serial.println(F("Z to initialize (zero) all\r "));
    Serial.println(F("W to save all to EEPROM\r "));
    say_quit();
    while((choice = get_char()) == 0x0d);
//    clear_serial_buffer();
//    while (!Serial.available());
//    choice = toupper(Serial.read());
    if(('0' <= choice)&&(choice <= '6')){
       Serial.print(F("Enter coefficient "));
       Serial.println(choice);
       get_to_sb();
       XC[choice-'0'] = atof(stringbuffer); 
    }else if(choice == 'Z'){
        for(int i = 0; i < 6; i++){
          XC[i] = 0.0;
        }
    }else if (choice == 'W'){
      for (int i = 0; i < 6; i++){
        C[i] = XC[i]; // copy locals to globals for use by program
        EEPROM_writeDouble(E_C0 + 4*i,C[i]); // each write takes 4 eeprom locations
      }
    }else{
      return;
    }	
  };
}


void get_poly_from_ee()
{
  for (int i = 0; i < 6; i++)
  {
    C[i] = EEPROM_readDouble(E_C0 + 4*i);
  }
}


void zero_poly() // initialize poly eeprom by setting all to 0
{
  for (int i = 0; i < 24; i++)
  {
    EEPROM.write(E_C0 + i, 0);
  }
}

// get_to_sb() below serves poly() above to read a string for a float
// into stringbuffer[].  Caller still has to use atof() to convert
// to float

void get_to_sb(){
  for (int i=0; i<21; i++) stringbuffer[i]=0; // make sure string ends in zero
  clear_serial_buffer();
  while (!Serial.available());
  if(Serial.peek() == 0x0D){
    Serial.read();
  }
  while (!Serial.available());
  Serial.readBytesUntil(0x0D, stringbuffer, 15);
}

void display_XC(float XC[]){
  for(int i = 0; i < 6; i++){
    Serial.print(i);
    Serial.print(F(" = "));
    Serial.print(dtostre(XC[i], stringbuffer, 15, 'E'));
    Serial.println(F("\r "));
  }
}




// Copied below functions to read / write double from
// http://forum.arduino.cc/index.php/topic,41497.0.html
// "Writing float values to Eeprom"  I couldn't get his
// "write anything" template to work, but modifying the
// below routines for other than double should be easy.



void EEPROM_writeDouble(int ee, double value)
{
  byte* p = (byte*)(void*)&value;
  for (int i = 0; i < sizeof(value); i++)
    EEPROM.write(ee++, *p++);
}

double EEPROM_readDouble(int ee)
{
  double value = 0.0;
  byte* p = (byte*)(void*)&value;
  for (int i = 0; i < sizeof(value); i++)
    *p++ = EEPROM.read(ee++);
  return value;
}

// ****************  SEND A FREQUENCY TO EEPROM ****************************

// NRK send the DDS clock frequency to EEPROM, low to high in adr 0 thru 3
// by Nick Kennedy WA5BDU
// 1r11 changed to use for any frequency, now you also pass the eeprom base adr

void f_to_ee(double f, int E_index){
  Serial.print(E_index);
  long i_f;
  i_f = (uint32_t) f;
  for (int i=E_index; i<(E_index + 4); i++){
    EEPROM.write(i, (byte) i_f & 0xFF);
    i_f >>= 8; // shift 1 byte to right
  }
}

// ****   READ FREQUENCY FROM EEPROM ***************************

// NRK get the stored DDS clock frequency from EEPROM for use here
// Rev 1r9: Changing to return value rather and set f_flock here ...
// Rev 1r11: This routine is now universal for freq from EE and user
// passes index to eeprom

double f_from_ee(int E_index){
  unsigned long the_freq = 0;
  for (int i=E_index+3; i>=E_index; i--){
    the_freq <<= 8;
    the_freq = the_freq | EEPROM.read(i);
  }
  return( (float) the_freq);
}    

// Get manual slope and intercept values from user:

void manual_slope_int(){
  char choice = '\0';
  
  Serial.print(F("Slope    : "));
  Serial.println(slope,3);
  Serial.print(F("Intercept: "));
  Serial.println(intercept,3);
  
  Serial.println(F("\rEnter slope as a decimal"));
  Serial.println(F("number (dBm / count): "));
  clear_serial_buffer();
  slope = Serial.parseFloat();
  Serial.println(slope,2);

  Serial.println(F("\rEnter intercept as a negative"));
  Serial.println(F("decimal number (dBm):"));
  clear_serial_buffer();
  intercept = Serial.parseFloat();
  Serial.println(intercept,2);	

  Serial.println(); // blank line
  say_press();
  Serial.println(F("Y to write to EEPROM\r "));
  Serial.println(F("N to cancel\r "));
  while (choice != 'N' && choice != 'Y'){
    while((choice = get_char()) == 0x0d);
    if (choice == 'N'){
      Serial.println(F("Function cancelled"));
      goto exit;
    }
  }
  slope_int_to_ee();
exit:;
}

// Below, save level through shorted test fixture to eeprom
// received as floating point.  Needs to be between -42 and +22
// dBm for storage as signed integer.  I multiply by 1000 and
// round to int.  To skew the range toward the negative, I add
// 10 to the received value and restore when reading back.

void loss_to_ee(float loss){
  unsigned int ee_loss;
  ee_loss = (unsigned int)(round((loss + 40.) * 1000.)); //   15 > loss > -40

  EEPROM.write(E_loss, (byte)(lowByte(ee_loss))); 
  EEPROM.write(E_loss + 1, (byte)(highByte(ee_loss)));
} 

// Below, retrieve loss value (level thru shorted fixture) from eeprom

float loss_from_ee(){
  unsigned int loss;
  unsigned hiloss;
  float floss;

  loss = (unsigned int) EEPROM.read(E_loss); // get lower byte
  hiloss = (unsigned int) (EEPROM.read(E_loss + 1));
  hiloss = hiloss<<8;
  loss = loss + hiloss;
  floss = (float) loss;
  return( floss/1000. - 40.);	

}


// Below, convert slope to integer slope times 100,000 and save to eeprom
// and convert intercept to positive integer times 100 and save to eeprom

void slope_int_to_ee(){
  float temp_floater;
  temp_floater = 100000.*slope; 

  i_slope = (unsigned int) round(temp_floater);
  EEPROM.write(E_slope, (byte)(lowByte(i_slope))); // saving slope* 100,000 as integer
  EEPROM.write(E_slope + 1, (byte)(highByte(i_slope)));

  temp_floater = abs(intercept) * 100.;
  i_intercept = (unsigned int) round(temp_floater);
  EEPROM.write(E_intercept, (byte) (lowByte(i_intercept))); // saving intercept * 100 as positive integer
  EEPROM.write(E_intercept + 1, (byte)(highByte(i_intercept))); // save high byte of word
} 

// Below, retrieve slope and intercept and convert back to floating point for use in program

void slope_int_from_ee(){
  i_slope = (unsigned int) EEPROM.read(E_slope); // get lower byte
  i_slope = i_slope + (unsigned int) (EEPROM.read(E_slope+1)<<8); // higher byte
  slope = ((float) i_slope)/100000.;

  i_intercept = (unsigned int) EEPROM.read(E_intercept); 
  i_intercept = i_intercept + (unsigned int) (EEPROM.read(E_intercept+1)<<8); 
  intercept = -1. * ((float) i_intercept)/100.;

  // Now see if retrieved values are acceptable:

  cal_flag = 0; // assume bad result
  if ( slope > 0.05 && slope < 0.2 && intercept > -100. && intercept < -70.){
    cal_flag = 1; // flag usable values
  }
}

/* REPLACE WITH ONE FUNCTION TO GET FREQ FROM EE
 
 // below su_from_ee() gets the start-up frequency from EEPROM:
 
 void su_from_ee(void) {
 long i_f_out = 0;
 
 for (int i=E_su_fq+3; i!=(E_su_fq-1); i--) {
 i_f_out <<= 8;
 i_f_out = i_f_out | EEPROM.read(i);
 }
 f_out = (double) i_f_out;
 }
 */

